/***************************************************************************/
/*  Copyright 1994 National Instruments Corporation.  All Rights Reserved. */
/***************************************************************************/
#include "\LW\include\vxi.h"
#include "\LW\include\gpib.h"
#include "\LW\include\formatio.h"
#include "\LW\include\lwsystem.h"
#include "eip123x.h"

/*= EIP 1230A/1231A VXIbus Pulse/CW Microwave Frequency Counters===========*/
/* LabWindows 2.0 Instrument Driver                                        */
/* Release: April 1994                                                     */
/* By: Jeff Orbach, LWILDP, NI Austin, TX  (800)433-3488                   */
/* Originally written in C                                                 */
/* Modification History:                                                   */
/* Modification Date:    10/93 Reviewed by EIP and minor changes were made.*/
/*                                                                         */
/* Modification Date:    04/94 Changed the GPIB Library calls and Word     */
/*                       Serial VXI Library calls so that they check for   */
/*                       the error bit being set to indicate an error      */
/*                       (& 0x8000) instead of checking for a negative     */
/*                       return value.                                     */
/*                                                                         */
/* Modified by:          Jeff Orbach, National Instruments, Austin, Texas  */
/*                       Phone (800) 433-3488  Fax (512) 794-5794          */
/*                                                                         */
/* Modification Date:    10/29/94 Added Low Level function class containing*/
/*                       write, read, reset, error_query, & revision_query.*/
/*                       Added _save_default_setup to Save/Recall class.   */
/*                       Added utilies read_file, write_file, poll, and    */
/*                       get_status_reg.                                   */
/*                       Improved help screens.                            */
/* Modified by:          Jim McDaniel for EIP Microwave Inc.               */
/*=========================================================================*/

#define eip123x_REVISION  "V1.2"

/*= UTILITY ROUTINES ======================================================*/
int eip123x_write (int, char* );
int eip123x_read (int, int, char*, long*);
int eip123x_reset (int);
int eip123x_error_query (int, int, int*, char*);
int eip123x_revision_query (int, char*, char*);
int eip123x_open_instr (int, int, int *);
int eip123x_close_instr (int);
int eip123x_init_gpib (int);
int eip123x_invalid_integer_range (int, int, int, int);
int eip123x_invalid_longint_range (long, long, long, int);
int eip123x_invalid_real_range (double, double, double, int);
int eip123x_device_closed (int);
int eip123x_read_data (int, char *, int);
int eip123x_write_data (int, char *, int);
int eip123x_read_data_file (int, char *);
int eip123x_write_data_file (int, char *);
int eip123x_read_reg (int, int, int *);
int eip123x_write_reg (int, int, int);
int eip123x_set_timeout (int, int);
int eip123x_poll (int, int *);
int eip123x_get_status_reg (int, int *);
void eip123x_setup_arrays (void);

/*= INSTRUMENT TABLES =====================================================*/
/* log_addr     contains the logical addresses of opened instruments.      */
/* bd           contains the device descriptors returned by OpenDev        */
/*              (NI-GPIB only)                                             */
/* interface    specifies the type of interface (NI-VXI or NI-GPIB).       */
/* base_addr    contains the base address in A16 space for the instr.      */
/* instr_cnt    contains the number of instruments open of this model type.*/
/* eip123x_err  the error variable for the instrument module               */
/*=========================================================================*/
static int  log_addr[eip123x_MAX_INSTR + 1];
static int  bd[eip123x_MAX_INSTR + 1];
static int  interface[eip123x_MAX_INSTR + 1];
static long base_addr[eip123x_MAX_INSTR + 1];
static int  instr_cnt;
static int  eip123x_err;
static int  model[eip123x_MAX_INSTR + 1];
static long bytes;

/*= STATIC DATA ===========================================================*/
/* cmd : a buffer for GPIB and VXI I/O strings.                            */
/* timeout: a look up table used to associate GPIB timeout codes with Word */
/*          Serial timout codes.                                           */
/*=========================================================================*/
static char cmd[50];
static long timeout[18];

/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/
/*=========================================================================*/
static char *units[4];
static char *hold[3];
static char *triger[2];
static char *output[13];
static char *measurement[3];
static char *special[4];
static char *count_reg_mode[2];
static char *ext_base_ref[2];
static double low_cen[9];
static double high_cen[9];

/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument, queries for ID, and       */
/*           initializes the instrument to a known state.                  */
/*=========================================================================*/
int eip123x_init (laddr, platform, id_query, rest, instrID, model_number)
int   laddr;
int   platform;
int   id_query;
int   rest;
int * instrID;
int * model_number;
{
    int id;
    int value;

    if (eip123x_invalid_integer_range (laddr, 0, 255, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (platform, 1, 2, -2) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (id_query, 0, 1, -3) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (rest, 0, 1, -4) != 0)
        return eip123x_err;

    /*   Initialize entry in Instrument Table and interface for instrument. */
    if (eip123x_open_instr (laddr, platform, &id) != 0)
        return eip123x_err;

    if (id_query) {
        if (eip123x_write_data (id, "OU ID", 5) != 0) {
            eip123x_close_instr (id);
            return eip123x_err;
        }
        if (eip123x_read_data (id, cmd, 50) != 0) {
            eip123x_close_instr (id);
            return eip123x_err;
        }
        Scan (cmd, "%s>EIP123%d[w1]", &model[id]);
        if (NumFmtdBytes () != 7 || model[id] < 0 || model[id] > 1) {
            eip123x_close_instr (id);
            *model_number = 1230 + model[id];
            eip123x_err = 223;
            return eip123x_err;
        }
    *model_number = 1230 + model[id];
    }

    /*   Initialize the instrument to a known state. */
    if (rest) {
        Fmt (cmd, "%s", "IN");
        if (eip123x_write_data (id, cmd, NumFmtdBytes ()) != 0)  {
            eip123x_close_instr (id);
            return eip123x_err;
        }
        delay (2.0);
    }

    /*  Initialize Instrument-Dependent Command arrays.  */
    eip123x_setup_arrays ();

    /*  Set timeout to 10 seconds  */
    if (eip123x_set_timeout (id, 13) != 0) {
        eip123x_close_instr (id);
        return eip123x_err;
    }
    *instrID = id;
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Bands for EIP1230 A                                           */
/* Purpose:  This Function contains the band configuration functions for   */
/*           the EIP1230A                                                  */
/*=========================================================================*/
int eip123x_band_eip1230 (instrID, band, subband)
int instrID;
int band;
int subband;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (band, 0, 3, -2) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (subband, 1, 8, -3) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;
    if (model[instrID] == 1)  {
        eip123x_err = 350;
        return eip123x_err;
    }
    if (band != 3 && subband != 1)  {
        eip123x_err = 330;
        return eip123x_err;
    }

    Fmt (cmd, "SU %d", subband);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.1);

    Fmt (cmd, "BA %d", band);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Bands for EIP1231 A                                           */
/* Purpose:  This Function contains the band configuration functions for   */
/*           the EIP1231A                                                  */
/*=========================================================================*/
int eip123x_band_eip1231 (instrID, band)
int instrID;
int band;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (band, 0, 2, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;
    if (!model[instrID])  {
        eip123x_err = 360;
        return eip123x_err;
    }

    Fmt (cmd, "BA %d", band);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Center Frequency                                              */
/* Purpose:  This function controls the center frequency mode of operation */
/*           in which the counter looks for a signal in the vicinity of    */
/*           the center frequency value.                                   */
/*=========================================================================*/
int eip123x_center_freq (instrID, center_freq)
int instrID;
double center_freq;
{
    int band;
    int sub_band;

    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_real_range (center_freq, 900.0, 170000.0, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "%s", "OU BA");
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    if (eip123x_read_data (instrID, cmd, 50) != 0)
        return eip123x_err;
    if (Scan (cmd, "%s>%d", &band) != 1)  {
        eip123x_err = 236;
        return eip123x_err;
    }
    if (eip123x_invalid_integer_range (band, 2, 3, 340) != 0)
        return eip123x_err;
    if (model[instrID] == 1)  {
        if (band != 2)  {
            eip123x_err = 340;
            return eip123x_err;
        }
        if (eip123x_invalid_real_range (center_freq, 900.0, 20500.0, -2) != 0
                && center_freq != 0.0)
            return eip123x_err;
    }
    if (!model[instrID])  {
        if (band == 2)
            if (eip123x_invalid_real_range (center_freq, 900.0, 26700.0, -2) != 0
                    && center_freq != 0.0)
                return eip123x_err;
        if (band == 3)  {
            if (eip123x_invalid_real_range (center_freq, 26500.0, 170050.0, -2) != 0
                    && center_freq != 0.0)
                return eip123x_err;
            Fmt (cmd, "%s", "OU SU");
            if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
                return eip123x_err;
            if (eip123x_read_data (instrID, cmd, 50) != 0)
                return eip123x_err;
            if (Scan (cmd, "%s>%d", &sub_band) != 1)  {
                eip123x_err = 236;
                return eip123x_err;
            }
            if (eip123x_invalid_integer_range (sub_band, 1, 8, 236) != 0)
                return eip123x_err;
            if (eip123x_invalid_real_range (center_freq, low_cen[sub_band], high_cen[sub_band], -2) != 0
                    && center_freq != 0.0)
                return eip123x_err;
        }
    }
    Fmt (cmd, "CE %f MHz", center_freq);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Frequency Window Limits                                       */
/* Purpose:  This function controls the high and low limits of the         */
/*           frequency window that is searched in Band 2.                  */
/*=========================================================================*/
int eip123x_freq_win_lim (instrID, lo_lim, hi_lim)
int instrID;
double lo_lim;
double hi_lim;
{
    int band;

    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_real_range (lo_lim, 900.0, 26700.0, -2) != 0)
        return eip123x_err;
    if (eip123x_invalid_real_range (hi_lim, lo_lim, 26700.0, -3) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "%s", "OU BA");
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    if (eip123x_read_data (instrID, cmd, 50) != 0)
        return eip123x_err;
    if (Scan (cmd, "%s>%d", &band) != 1)  {
        eip123x_err = 236;
        return eip123x_err;
    }
    if (band != 2)  {
        eip123x_err = 380;
        return eip123x_err;
    }
    if (model[instrID] == 1)  {
        if (eip123x_invalid_real_range (lo_lim, 900.0, 20500.0, -2) != 0)
            return eip123x_err;
        if (eip123x_invalid_real_range (hi_lim, lo_lim, 20500.0, -3) != 0)
            return eip123x_err;
    }

    Fmt (cmd, "LO %f MHZ, HI %f MHZ", lo_lim, hi_lim);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Sample Rate                                                   */
/* Purpose:  This Function sets a delay between measurement values (0 to   */
/*           100 S to a 10 mS resolution).                                 */
/*=========================================================================*/
int eip123x_sample_rate (instrID, sample_rate)
int instrID;
long sample_rate;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_longint_range (sample_rate, 0L, 100000L, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "SA %i[b4] MS", sample_rate);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Resolution                                                    */
/* Purpose:  This Function controls the frequency measurement resolution.  */
/*=========================================================================*/
int eip123x_resolution (instrID, res)
int instrID;
int res;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (res, 2, 9, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "RE %d", res);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Frequency Average                                             */
/* Purpose:  This Function controls the number of measurements to be       */
/*           averaged on frequency and pulse measurements.                 */
/*=========================================================================*/
int eip123x_freq_avr (instrID, freq_avr)
int instrID;
int freq_avr;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (freq_avr, 1, 99, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "AV %d", freq_avr);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Frequency Offset                                              */
/* Purpose:  This Function allows the entry of a positive or negative      */
/*           frequency to 1 kHz resolution into the offset frequency       */
/*           register.                                                     */
/*=========================================================================*/
int eip123x_freq_offset (instrID, multiplier, off_freq)
int instrID;
int multiplier;
double off_freq;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (multiplier, 1, 99, -2) != 0)
        return eip123x_err;
    if (eip123x_invalid_real_range (off_freq, -99.99999, 99.99999, -3) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "MU %d, OF %f GHZ", multiplier, off_freq);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Minimum Pulse Repetition                                      */
/* Purpose:  This Function controls the minimum pulse repetition           */
/*           frequency of the pulsed signals that can be acquired and      */
/*           measured by the counter.                                      */
/*=========================================================================*/
int eip123x_min_freq_rep (instrID, min_prf, num_unit)
int instrID;
double min_prf;
int num_unit;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (num_unit, 0, 3, -3) != 0)
        return eip123x_err;
    switch (num_unit)  {
        case 0:
            if (eip123x_invalid_real_range (min_prf, 0.0, 1.7e11, -2) != 0)
                return eip123x_err;
            break;
        case 1:
            if (eip123x_invalid_real_range (min_prf, 0.0, 170000000.0, -2) != 0)
                return eip123x_err;
            break;
        case 2:
            if (eip123x_invalid_real_range (min_prf, 0.0, 170000.0, -2) != 0)
                return eip123x_err;
            break;
        case 3:
            if (eip123x_invalid_real_range (min_prf, 0.0, 170.0, -2) != 0)
                return eip123x_err;
            break;
    }
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "MI %f %s", min_prf, units[num_unit]);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Inhibit Input                                                 */
/* Purpose:  This Function selects the source of the signal controlling    */
/*           when the counter can detected a signal.                       */
/*=========================================================================*/
int eip123x_inhibit_input (instrID, inhib_inp)
int instrID;
int inhib_inp;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (inhib_inp, 0, 2, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "INHI %d", inhib_inp);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Time Base Source                                              */
/* Purpose:  This Function selects the source of the 10 MHz frequency      */
/*           reference signal. This function was added by Chuck McDaniel   */
/*           of EIP on 9/30/93                                             */
/*=========================================================================*/
int eip123x_time_base (instrID, ref)
int instrID;
int ref;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (ref, 0, 1, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "SP %s", ext_base_ref[ref]);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Trigger Set Up                                                */
/* Purpose:  This Function selects the triggering mode, configures the     */
/*           INT/EXT time base reference, and selects the TTLTRG line pair */
/*           to be used in the Asynchronous Trigger Protocol.              */
/*=========================================================================*/
int eip123x_trig_setup (instrID, synch_line, asynch_line)
int instrID;
int synch_line;
int asynch_line;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (synch_line, 0, 9, -2) != 0 || synch_line == 8) {
        eip123x_err = -2;
        return eip123x_err;
    }
    if (eip123x_invalid_integer_range (asynch_line, 0, 9, -3) != 0 || synch_line == 8) {
        eip123x_err = -3;
        return eip123x_err;
    }
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "SY %d, AS %d", synch_line, asynch_line);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Special Functions                                             */
/* Purpose:  This Function selects aspecial function from the special      */
/*           functions which this driver supports.                         */
/*=========================================================================*/
int eip123x_special_funct (instrID, special_select, return_val)
int instrID;
int special_select;
double *return_val;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (special_select, 0, 2, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "%s", special[special_select]);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 50) != 0)
        return eip123x_err;
    if (Scan (cmd, "%s>%f", return_val) != 1)  {
        eip123x_err = 236;
        return eip123x_err;
    }

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Terminate All Specials                                        */
/* Purpose:  This Function terminates all previously implemented special   */
/*           functions.                                                    */
/*=========================================================================*/
int eip123x_terminate_spec (instrID)
int instrID;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "%s", "OF 0, SP 00");
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);
    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Hold Function                                                 */
/* Purpose:  This Function selects the status of the Hold command.         */
/*=========================================================================*/
int eip123x_hold_func (instrID, hol)
int instrID;
int hol;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (hol, 0, 1, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    if (!hol)  {
        /*   Hold OFF */
        Fmt (cmd, "HO %s, OU BA", hold[hol]);
        if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return eip123x_err;
        delay (0.2);
        if (eip123x_read_data (instrID, cmd, 30) != 0)
            return eip123x_err;
        if (eip123x_read_data (instrID, cmd, 30) != 0)
            return eip123x_err;
    }
    else  {
        Fmt (cmd, "HO %s, OU FR", hold[hol]);
        if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
            return eip123x_err;
        delay (0.2);
        if (eip123x_read_data (instrID, cmd, 30) != 0)
            return eip123x_err;
    }

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Set Measurement Type                                          */
/* Purpose:  This Function selects which type of measurement will be       */
/*           taken, Frequency, Width, or Period.                           */
/*=========================================================================*/
int eip123x_set_meas_type (instrID, meas_type)
int instrID;
int meas_type;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (meas_type, 0, 3, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "%s", measurement[meas_type]);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Trigger Function                                              */
/* Purpose:  This Function selects and implements one of the two internal  */
/*           trigger functions, trigger or reset.                          */
/*=========================================================================*/
int eip123x_trigger_func (instrID, trig)
int instrID;
int trig;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (trig, 0, 1, -3) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "%s", triger[trig]);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    delay (0.2);

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Get Measurement                                               */
/* Purpose:  This Function reads the measurement from the instrument.      */
/*=========================================================================*/
int eip123x_get_measurement (instrID, result)
int instrID;
double *result;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    if (eip123x_read_data (instrID, cmd, 30) != 0)
        return eip123x_err;
    if (Scan (cmd, "%s>%f", result) != 1)  {
        eip123x_err = 236;
        return eip123x_err;
    }

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Save/Recall Instr Setup                                       */
/* Purpose:  This Function selects and implements the Store or Recall      */
/*           functions.                                                    */
/*=========================================================================*/
int eip123x_counter_setup (instrID, mod, reg)
int instrID;
int mod;
int reg;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (mod, 0, 2, -2) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (reg, 0, 9, -3) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;
    Fmt (cmd, "%s %d", count_reg_mode[mod], reg);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    return eip123x_err;
}
/*=========================================================================*/
/* Function: save_default_setup                                            */
/* Purpose:  Restores save/recall register 0 to contain the factory        */
/*           default instrument setup.                                     */
/*=========================================================================*/
int eip123x_save_default_setup (instrID)
int instrID;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;
    Fmt (cmd, "%s", "ST DEFAULT");
    eip123x_write_data (instrID, cmd, NumFmtdBytes ());
    return eip123x_err;
}
/*=========================================================================*/
/* Function: Get General Setup Info                                        */
/* Purpose:  This Function returns the current status of the selected.     */
/*           parameter                                                     */
/*=========================================================================*/
int eip123x_get_gen_setup_info (instrID, setup, display)
int instrID;
int setup;
char *display;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (setup, 0, 12, -2) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "%s", output[setup]);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    if (eip123x_read_data (instrID, cmd, 20) != 0)
        return eip123x_err;
    if (Scan (cmd, "%s>%s", display) != 1)  {
        eip123x_err = 236;
        return eip123x_err;
    }
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int eip123x_close (instrID)
int instrID;
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_device_closed (instrID) != 0)
        return eip123x_err;

    eip123x_close_instr (instrID);

    return eip123x_err;
}

/*====== UTILITY ROUTINES =================================================*/

/*=========================================================================*/
/* Function: Write To Instrument                                           */
/* Purpose:  This function writes a command string to the instrument.      */
/*=========================================================================*/
int eip123x_write (int instrID, char *cmd_string)
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_device_closed(instrID) != 0)
        return eip123x_err;

    Fmt (cmd, "%s<%s", cmd_string);
    if (eip123x_write_data (instrID, cmd, NumFmtdBytes()) != 0)
        return eip123x_err;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Read Instrument Buffer                                        */
/* Purpose:  This function reads the output buffer of the instrument       */
/*           This function checks that the read is valid and returns the   */
/*           actual number of characters read. The user should use this    */
/*           function rather than _read_data which it calls.               */
/* Input:    int instrID                                                   */
/*           int numbytes      maximum number of bytes to be read          */
/*           char *in_buff     address of character buffer to read into    */
/* Output:   long *bytes_read  actual number of bytes read                 */
/*           returns           eip123x_err                                 */
/* Calls:    eip123x_read_data                                             */
/*=========================================================================*/
int eip123x_read (int instrID, int numbytes, char *in_buff, long *bytes_read)
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_device_closed(instrID) != 0)
        return eip123x_err;

    *bytes_read = 0L;
    if (eip123x_read_data (instrID, in_buff, numbytes) != 0)
        return eip123x_err;

    switch (interface[instrID]) {
        case NI_VXI: *bytes_read = bytes;
                     break;
       case NI_GPIB: *bytes_read = (long)ibcnt;
                     break;
            default: *bytes_read = 0L;
                     break;
    }

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Reset                                                         */
/* Purpose:  This function resets the instrument.                          */
/*=========================================================================*/
int eip123x_reset (int instrID)
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_device_closed(instrID) != 0)
        return eip123x_err;
    if (eip123x_write_data (instrID, "IN", 2) != 0)
        return eip123x_err;
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Error Query                                                   */
/* Purpose:  This function either queries the instrument error queue,      */
/*           or clears it.                                                 */
/*=========================================================================*/
int eip123x_error_query (int instrID, int action, int *error, char *error_message)
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_invalid_integer_range (action, 0, 1, -1) != 0)
        return eip123x_err;
    if (eip123x_device_closed(instrID) != 0)
        return eip123x_err;

    Fmt (error_message, "%s<%s", "");
    if (action == 0) {
        if (eip123x_write_data (instrID, "OU ER", 5) != 0)
            return eip123x_err;
        if (eip123x_read_data (instrID, cmd, 50) != 0)
            return eip123x_err;
        if (Scan (cmd, "%s>%i", error) != 1) {
            eip123x_err = 236;
            return eip123x_err;
        }
    }
    else {
        eip123x_write_data (instrID, "CL", 2);
    }
    return eip123x_err;
}
/*=========================================================================*/
/* Function: Revision                                                      */
/* Purpose:  This function returns the driver and instrument revisions.    */
/*=========================================================================*/
int eip123x_revision_query (int instrID, char *driver_rev, char *instr_rev)
{
    if (eip123x_invalid_integer_range (instrID, 1, eip123x_MAX_INSTR, -1) != 0)
        return eip123x_err;
    if (eip123x_device_closed(instrID) != 0)
        return eip123x_err;
    Fmt (driver_rev, "%s<%s", eip123x_REVISION);
    if (eip123x_write_data (instrID, "OU DA", 5) != 0)
        return eip123x_err;
    eip123x_read_data (instrID, instr_rev, 40);
    return eip123x_err;
}

/******** OLD FUNCTIONS **********************************/
/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table.  The variable platform equals 1 for NI-VXI  */
/*           and 2 for NI-GPIB. The size of the Instrument Table can be    */
/*           changed in the include file by altering the constant          */
/*           eip123x_MAX_INSTR. The return value of this function is equal */
/*           to the global error variable.                                 */
/*=========================================================================*/
int eip123x_open_instr (laddr, platform, id)
int laddr;
int platform;
int *id;
{
    int i;
    int instrID;

    instrID = 0;
    eip123x_err = 0;
    /*   Check to see if the instrument is already in the Instrument Table. */
    for (i = 1; i <= eip123x_MAX_INSTR; i++)
        if (log_addr[i] == laddr)  {
            instrID = i;
            i = eip123x_MAX_INSTR;
        }
    /*   If it is not in the instrument table, open an entry for the instrument. */
    if (instrID <= 0)
        for (i = 1; i <= eip123x_MAX_INSTR; i++)
            if (!log_addr[i])  {
                instrID = i;
                log_addr[instrID] = laddr;
                base_addr[instrID] = (long)(laddr * 0x40) + 0xc000L;
                interface[instrID] = platform;
                instr_cnt++;
                i = eip123x_MAX_INSTR;
            }
    /*   If an entry could not be opened in the Instrument Table, return an error.*/
    if (instrID <= 0)  {
        eip123x_err = 220;
        return eip123x_err;
    }
    /*   Initialize the interface. */
    switch (platform)  {
    case NI_VXI:
        if (InitVXIlibrary () < 0)  {
            eip123x_close_instr (instrID);
            eip123x_err = 222;
            return eip123x_err;
        }
        break;
    case NI_GPIB:
        if (eip123x_init_gpib (instrID) != 0)  {
            eip123x_close_instr (instrID);
            return eip123x_err;
        }
        break;
    }
    *id = instrID;
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table, if applicable (NI-GPIB only), and setting  */
/*           the logical address, bd, and interface to zero in the         */
/*           Instrument Table. Upon successful completion, zero is         */
/*           returned, otherwise -1 is returned.                           */
/*=========================================================================*/
int eip123x_close_instr (instrID)
int instrID;
{
    if (interface[instrID] == NI_GPIB)
        if (bd[instrID] > 0)
            if (CloseDev (bd[instrID]) < 0)  {
                eip123x_err = 221;
                return eip123x_err;
            }
    log_addr[instrID] = 0;
    bd[instrID] = 0;
    base_addr[instrID] = 0L;
    interface[instrID] = 0;
    model[instrID] = 0;
    instr_cnt -= 1;

    return eip123x_err;
}

/*=========================================================================*/
/*                         NI-GPIB ONLY                                    */
/* Function: Initialize GPIB                                               */
/* Purpose:  This function initializes the GPIB interface.  First, the     */
/*           GPIB-VXI interface is initialized, then the Instrument        */
/*           interface is initialized.  The primary address of all entries */
/*           is set to 1.  The secondary address the GPIB-VXI is 0 and the */
/*           secondary address of the instrument is obtained by querying   */
/*           the GPIB-VXI.  The return value of the function is equal to   */
/*           the global error variable.                                    */
/*=========================================================================*/
int eip123x_init_gpib (instrID)
int instrID;
{
    int sec_addr;

    eip123x_err = 0;
  /*   If the GPIB-VXI is not opened (bd[0]=0), open it and initialize its */
  /*   primary and secondary addresses.                                    */
    if (bd[0] <= 0)  {
        CloseInstrDevs ("eip123x");
        /*   Set the interface of the GPIB-VXI.*/
        interface[0] = NI_GPIB;
        bd[0] = OpenDev ("", "eip123x");
        if (bd[0] <= 0)  {
            eip123x_err = 220;
            return eip123x_err;
        }
        if (ibpad (bd[0], 1) & 0x8000)  {
            eip123x_err = 233;
            return eip123x_err;
        }
        if (ibsad (bd[0], 0x60) & 0x8000)  {
            eip123x_err = 233;
            return eip123x_err;
        }
    }
    /*   If the instrument is not opened (bd[instrID]=0), open it and initialize */
    /*   its primary and secondary addresses.                                    */
    if (bd[instrID] <= 0)  {
        bd[instrID] = OpenDev ("", "eip123x");
        if (bd[instrID] <= 0)  {
            eip123x_err = 220;
            return eip123x_err;
        }
    }
    /*   Query the GPIB-VXI for the secondary address of the instrument. */
    Fmt (cmd, "%s<LaSaddr? %d", log_addr[instrID]);
    if (eip123x_write_data (0, cmd, NumFmtdBytes ()) != 0)
        return eip123x_err;
    if (eip123x_read_data (0, cmd, 50) != 0)
        return eip123x_err;
    if (Scan (cmd, "%s>%d", &sec_addr) != 1)  {
        eip123x_err = 236;
        return eip123x_err;
    }
    if (ibpad (bd[instrID], 1) & 0x8000)  {
        eip123x_err = 233;
        return eip123x_err;
    }
    if (sec_addr != 255)  {
        if (ibsad (bd[instrID], sec_addr + 0x60) & 0x8000)  {
            eip123x_err = 233;
            return eip123x_err;
        }
    }
    else  if (ibsad (bd[instrID], 0x60) & 0x8000)  {
        eip123x_err = 233;
        return eip123x_err;
    }
    if (ibconfig (bd[instrID], 6, 1) & 0x8000)  {
        eip123x_err = 241;
        return eip123x_err;
    }
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
int eip123x_invalid_integer_range (val, min, max, err_code)
int val;
int min;
int max;
int err_code;
{

    if (val < min || val > max)
        eip123x_err = err_code;
    else
        eip123x_err = 0;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Invalid Long Integer Range                                    */
/* Purpose:  This function checks a long integer to see if it lies between */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
int eip123x_invalid_longint_range (val, min, max, err_code)
long val;
long min;
long max;
int err_code;
{

    if (val < min || val > max)
        eip123x_err = err_code;
    else
        eip123x_err = 0;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
int eip123x_invalid_real_range (val, min, max, err_code)
double val;
double min;
double max;
int err_code;
{

    if (val < min || val > max)
        eip123x_err = err_code;
    else
        eip123x_err = 0;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the instrument has been        */
/*           initialized. The return value is equal to the global error    */
/*           variable.                                                     */
/*=========================================================================*/
int eip123x_device_closed (instrID)
int instrID;
{

    if (log_addr[instrID] <= 0)
        eip123x_err = 232;
    else
        eip123x_err = 0;

    return eip123x_err;
}

/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int eip123x_read_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{

    eip123x_err = 0;
    switch (interface[instrID])  {
    case NI_VXI:
        if (WSrd (log_addr[instrID], buf, (long)cnt, 1, &bytes) & 0x8000)
            eip123x_err = 231;
        break;
    case NI_GPIB:
        if (ibrd (bd[instrID], buf, (long)cnt) & 0x8000)
            eip123x_err = 231;
        break;
    default:
        eip123x_err = 231;
        break;
    }
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int eip123x_write_data (instrID, buf, cnt)
int   instrID;
char *buf;
int   cnt;
{

    eip123x_err = 0;
    switch (interface[instrID])  {
    case NI_VXI:
        if ((WSwrt (log_addr[instrID], buf, (long)cnt, 3, &bytes) & 0x8000) || (bytes != (long)cnt))
            eip123x_err = 230;
        break;
    case NI_GPIB:
        if (ibwrt (bd[instrID], buf, (long)cnt) & 0x8000)
            eip123x_err = 230;
        break;
    default:
        eip123x_err = 230;
        break;
    }
    return eip123x_err;
}
/*=========================================================================*/
/* Function: Read Data File                                                */
/* Purpose:  This function reads a buffer of data from the instrument and  */
/*           stores it to the file specified by "filename".                */
/* Inputs:   int instrID and char *filename.                               */
/*           Filename must either be a string, such as "C:\lw\instr\file"  */
/*           or a pointer to such a string.  The return value is equal     */
/*           to the global                                                 */
/*           error variable.                                               */
/* Bewares:  This function does not first check that the read is valid.    */
/*=========================================================================*/
int eip123x_read_data_file (int instrID, char *filename)
{
    eip123x_err = 0;

    switch (interface[instrID]) {
        case NI_VXI: if (WSrdf (log_addr[instrID], filename, -1L, 1, &bytes) & 0x8000)
                        eip123x_err = 229;
                     break;
       case NI_GPIB: if (ibrdf (bd[instrID], filename) & 0x8000)
                        eip123x_err = 229;
                     break;
            default: eip123x_err = 229;
                     break;
    }
   return eip123x_err;
}

/*=========================================================================*/
/* Function: Write Data File                                               */
/* Purpose:  This function writes a buffer of data from the file specified */
/*           by "filename" to the instrument. Filename must either be a    */
/*           string, such as "C:\lw\instr\file" or a pointer to such a     */
/*           string.  The return value is equal to the global error        */
/*           variable.                                                     */
/*=========================================================================*/
int eip123x_write_data_file (int instrID, char *filename)
{
    eip123x_err = 0;

    switch (interface[instrID]) {
        case NI_VXI: if (WSwrtf (log_addr[instrID], filename, -1L, 1, &bytes) & 0x8000)
                        eip123x_err = 228;
                     break;
       case NI_GPIB: if (ibwrtf (bd[instrID], filename) & 0x8000)
                        eip123x_err = 228;
                     break;
            default: eip123x_err = 228;
                     break;
    }
    return eip123x_err;
}


/*=========================================================================*/
/* Function: Read Register                                                 */
/* Purpose:  This function reads an instrument register.  The return value */
/*           is equal to the global error variable.                        */
/*=========================================================================*/
int eip123x_read_reg (instrID, rel_addr, value)
int instrID;
int rel_addr;
int *value;
{
    eip123x_err = 0;

    switch (interface[instrID])  {
    case NI_VXI:
        if (VXIinReg (log_addr[instrID], rel_addr, value) < 0)
            eip123x_err = 231;
        break;
    case NI_GPIB:
        Fmt (cmd, "RREG? %d,%d", log_addr[instrID], rel_addr);
        if (ibwrt (bd[0], cmd, (long)NumFmtdBytes ()) & 0x8000)
            eip123x_err = 230;
        if (ibrd (bd[0], cmd, 20L) & 0x8000)
            eip123x_err = 231;
        if (Scan (cmd, "%s>%x", value) != 1)
            eip123x_err = 236;
        break;
    default:
        eip123x_err = 231;
        break;
    }
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Write Register                                                */
/* Purpose:  This function writes an integer value to an instrument        */
/*           register.  The return value is equal to the global error      */
/*           variable.                                                     */
/*=========================================================================*/
int eip123x_write_reg (instrID, rel_addr, value)
int instrID;
int rel_addr;
int value;
{
    eip123x_err = 0;

    switch (interface[instrID])  {
    case NI_VXI:
        if (VXIoutReg (log_addr[instrID], rel_addr, value) < 0)
            eip123x_err = 230;
        break;
    case NI_GPIB:
        Fmt (cmd, "WREG %d,%d,%d[u]", log_addr[instrID], rel_addr, value);
        if (ibwrt (bd[0], cmd, (long)NumFmtdBytes ()) & 0x8000)
            eip123x_err = 230;
        break;
    default:
        eip123x_err = 230;
        break;
    }
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Set Timeout                                                   */
/* Purpose:  This function changes or disables the timeout of the device.  */
/*           Refer to array at the end of this file for timeout codes      */
/*           (index of timeout array).                                     */
/*=========================================================================*/
int eip123x_set_timeout (instrID, tmo_code)
int instrID;
int tmo_code;
{
    long actual_tmo;

    eip123x_err = 0;
    switch (interface[instrID])  {
    case NI_VXI:
        if (WSsetTmo (timeout[tmo_code], &actual_tmo) < 0)
            eip123x_err = 239;
        break;
    case NI_GPIB:
        if (ibtmo (bd[instrID], tmo_code) & 0x8000)
            eip123x_err = 239;
        break;
    default:
        eip123x_err = 239;
        break;
    }
    return eip123x_err;
}
/*=========================================================================*/
/* Function: Poll                                                          */
/* Purpose:  This function performs a serial poll on the instrument.  The  */
/*           status byte of the instrument is placed in the response       */
/*           variable. The return value is the global error variable.      */
/*=========================================================================*/
int eip123x_poll (int instrID, int *response)
{
    eip123x_err = 0;

    switch (interface[instrID]) {
        case NI_VXI: if (WScmd (log_addr[instrID], 0xcfff, 1, response) & 0x8000)
                        eip123x_err = 226;
                     break;
       case NI_GPIB: if (ibrsp (bd[instrID], response) & 0x8000)
                        eip123x_err = 226;
                     break;
            default: eip123x_err = 226;
                     break;
    }
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Get Status Register                                           */
/* Purpose:  This function reads the VXI status register of the instrument */
/*           which is located at the instrument's base_addr + 4.           */
/*           Note that this is not the same value read by the poll function*/
/*           The return value is the global error variable.                */
/*=========================================================================*/
int eip123x_get_status_reg (int instrID, int *response)
{
    if (eip123x_read_reg (instrID, 4, response) != 0)
        eip123x_err = 226;
    else
        eip123x_err = 0;
    return eip123x_err;
}

/*=========================================================================*/
/* Function: Setup Arrays                                                  */
/* Purpose:  This function is called by the init routine to initialize     */
/*           static arrays.  This routine should be modified for each      */
/*           instrument to include instrument dependent command arrays.    */
/*=========================================================================*/
void eip123x_setup_arrays (void)
{

    /*   Timeout array values are equivalent to timeout in milliseconds. */
    timeout[0] = 0xffffffffL; /*   Equivalent to turning timeout off */
    timeout[1] = 1L;
    timeout[2] = 1L;
    timeout[3] = 1L;
    timeout[4] = 1L;
    timeout[5] = 1L;
    timeout[6] = 3L;
    timeout[7] = 10L;
    timeout[8] = 30L;
    timeout[9] = 100L;
    timeout[10] = 300L;
    timeout[11] = 1000L;
    timeout[12] = 3000L;
    timeout[13] = 10000L;
    timeout[14] = 30000L;
    timeout[15] = 100000L;
    timeout[16] = 300000L;
    timeout[17] = 1000000L;
    output[0] = "OU AV";
    output[1] = "OU BA";
    output[2] = "OU SU";
    output[3] = "OU CE";
    output[4] = "OU RE";
    output[5] = "OU LO";
    output[6] = "OU HI";
    output[7] = "OU AS";
    output[8] = "OU SY";
    output[9] = "OU INHI";
    output[10] = "OU MI";
    output[11] = "OU MU";
    output[12] = "OU OF";

    count_reg_mode[0] = "FE";
    count_reg_mode[1] = "ST";

    hold[0] = "OFF";
    hold[1] = "On";

    units[0] = "Hz";
    units[1] = "kHz";
    units[2] = "MHz";
    units[3] = "GHz";

    measurement[0] = "OU FR";
    measurement[1] = "OU WI";
    measurement[2] = "OU PE";

    ext_base_ref[0] = " 09";
    ext_base_ref[1] = " 08";

    triger[0] = "RESE";
    triger[1] = "TR";

    special[0] = "SP 01";
    special[1] = "SP 74";
    special[2] = "SP 75";

    low_cen[1] = 26500.0;
    low_cen[2] = 33000.0;
    low_cen[3] = 40000.0;
    low_cen[4] = 50000.0;
    low_cen[5] = 60000.0;
    low_cen[6] = 75000.0;
    low_cen[7] = 90000.0;
    low_cen[8] = 110000.0;

    high_cen[1] = 40000.0;
    high_cen[2] = 50000.0;
    high_cen[3] = 60000.0;
    high_cen[4] = 75000.0;
    high_cen[5] = 90000.0;
    high_cen[6] = 110000.0;
    high_cen[7] = 140000.0;
    high_cen[8] = 170000.0;
}

/*=== THE END =============================================================*/
